/* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR REPRESENTATIONS,
 * EITHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING ANY IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY
 * OR COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE. TI DISCLAIMS
 * ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET POSSESSION, AND NON-INFRINGEMENT
 * OF ANY THIRD PARTY INTELLECTUAL PROPERTY RIGHTS WITH REGARD TO THE PROGRAM OR
 * YOUR USE OF THE PROGRAM.
 * IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL, CONSEQUENTIAL OR
 * INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY THEORY OF LIABILITY AND WHETHER OR
 * NOT TI HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY
 * OUT OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM.  EXCLUDED
 * DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF REMOVAL OR REINSTALLATION,
 * COMPUTER TIME, LABOR COSTS, LOSS OF GOODWILL, LOSS OF PROFITS, LOSS OF
 * SAVINGS, OR LOSS OF USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S
 * AGGREGATE LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF YOUR USE OF THE
 * PROGRAM EXCEED FIVE HUNDRED DOLLARS (U.S.$500).
 * Unless otherwise stated, the Program written and copyrighted by Texas
 * Instruments is distributed as "freeware".  You may, only under TI's copyright
 * in the Program, use and modify the Program without any charge or restriction.
 * You may distribute to third parties, provided that you transfer a copy of this
 * license to the third party and the third party agrees to these terms by its
 * first use of the Program. You must reproduce the copyright notice and any
 * other legend of ownership on each copy or partial copy, of the Program.
 * You acknowledge and agree that the Program contains copyrighted material,
 * trade secrets and other TI proprietary information and is protected by
 * copyright laws, international copyright treaties, and trade secret laws, as
 * well as other intellectual property laws.  To protect TI's rights in the
 * Program, you agree not to decompile, reverse engineer, disassemble or
 * otherwise translate any object code versions of the Program to a
 * human-readable form.  You agree that in no event will you alter, remove or
 * destroy any copyright notice included in the Program.  TI reserves all rights
 * not specifically granted under this license. Except as specifically provided
 * herein, nothing in this agreement shall be construed as conferring by
 * implication, estoppel, or otherwise, upon you, any license or other right
 * under any TI patents, copyrights or trade secrets.
 * You may not use the Program in non-TI devices. */


//******************************************************************************
//
//  adc.c
//  Functions for using the integrated AD converters
//  The three AD inputs are used for the NTCs (temperature measurement)
//
//  Design Services - EMEA Power Management
//  Texas Instruments, Inc.
//
//******************************************************************************
// ANALOG CHANNELS:
// A0 (P2.0):	Temperature
// A3 (P2.3):	Input Vac voltage
// A6 (P3.6):	Vout
// A7 (P3.7):	Iout


#include "main.h"



// initialization of analog inputs
// temperature measurements with LMT87 temperature sensor, output voltage and current, pre-charge voltage, input Vac
void init_adc(void)
{
	// select internal reference and GND
	clear_bit(ADC10CTL0, SREF2);
	clear_bit(ADC10CTL0, SREF1);
	set_bit(ADC10CTL0, SREF0);
	
	// set sample-and-hold time to 16 ADC10CLKs
	set_bit(ADC10CTL0, ADC10SHT1);
	clear_bit(ADC10CTL0, ADC10SHT0);
	
	// set the sample rate to 200 ksps
	clear_bit(ADC10CTL0, ADC10SR);
	
	// no reference output
	clear_bit(ADC10CTL0, REFOUT);
	
	// reference always buffered
	clear_bit(ADC10CTL0, REFBURST);
	
	// single sample and conversion
	clear_bit(ADC10CTL0, MSC);
	
	// set reference generator voltage to 2.5V
	set_bit(ADC10CTL0, REF2_5V);
	
	// switch reference generator on
	set_bit(ADC10CTL0, REFON);
	
	// switch ADC on
	set_bit(ADC10CTL0, ADC10ON);
	
	// no interrupt
	clear_bit(ADC10CTL0, ADC10IE);
	
	// select A0 as default
	clear_bit(ADC10CTL1, INCH3);
	clear_bit(ADC10CTL1, INCH2);
	clear_bit(ADC10CTL1, INCH1);
	clear_bit(ADC10CTL1, INCH0);
	
	// ADC10SC bit as sample-and-hold source
	clear_bit(ADC10CTL1, SHS1);
	clear_bit(ADC10CTL1, SHS0);
	
	// ADC10 data as straight binary
	clear_bit(ADC10CTL1, ADC10DF);
	
	// sample-and-hold signal not inverted
	clear_bit(ADC10CTL1, ISSH);
	
	// set clock divider to 8 (2 MHz)
	set_bit(ADC10CTL1, ADC10DIV2);
	set_bit(ADC10CTL1, ADC10DIV1);
	set_bit(ADC10CTL1, ADC10DIV0);
	
	// MCLK as source for ADC
	set_bit(ADC10CTL1, ADC10SSEL1);
	clear_bit(ADC10CTL1, ADC10SSEL0);
	
	// convert a single channel once
	clear_bit(ADC10CTL1, CONSEQ1);
	clear_bit(ADC10CTL1, CONSEQ0);
	
	
	// setup of inputs
	// set pins as inputs
	clear_bit(A0_PDIR, A0_BIT);
	clear_bit(A15_PDIR, A15_BIT);
	clear_bit(A6_PDIR, A6_BIT);
	clear_bit(A7_PDIR, A7_BIT);
	
	// disable pull ups
	clear_bit(A0_RES, A0_BIT);
	clear_bit(A15_RES, A15_BIT);
	clear_bit(A6_RES, A6_BIT);
	clear_bit(A7_RES, A7_BIT);
	
	// enable analog input for A0
	clear_bit(ADC10AE0, A7_BIT);
	clear_bit(ADC10AE0, A6_BIT);
	clear_bit(ADC10AE1, A15_BIT);
	set_bit(ADC10AE0, A0_BIT);
}



// Conversion of the sampled digital value into degrees.
// The minimum value is 10 degree. For a temperature below a
// value of 0 is returned.
// The maximum value is 100 degree. For a temperature above a
// value of 200 is returned.
unsigned int lsb_to_temperature_adc(unsigned int lsb)
{
	unsigned int degree_plus_50=0;
	unsigned long temperature_long;
	
	// The output voltage of the LMT87 sensor is attenuated by a voltage divider 47.5K / 169K
	// Since the temperature can be negative, an offset of +50C has been added
	// This formula is used to calculate the temperature from the LSBs:
	// degree_plus50 = 244 - LSB * 0.2372
	// To prevent an overflow, the temperature is clamped to -50C
	
	temperature_long = (unsigned long)lsb * 2372;
	temperature_long = temperature_long / 10000;
	if (temperature_long < 244) temperature_long = 244 - temperature_long;
	else temperature_long = 0;
	degree_plus_50 = temperature_long;
	return degree_plus_50;
}



// Conversion of the sampled digital value into voltage.
// The value has three digits, the last represents hundred millivolts.
// (1358 means 13.58V)
unsigned int lsb_to_voltage_adc(unsigned int lsb)
{
	unsigned long voltage_long;
	unsigned int voltage_int;
	
	// The output voltage is attenuated by a voltage divider 60.4K / 4.75K.
	// This formula is used to calculate the output voltage from the LSBs:
	// With 2.5V (LSB=1023) present on the ADC, the resistor divider is set for a full scale of 34.3V (equivalent to 343)
	// Vout = LSB * (2.5/1023) * (60.4K + 4.75K) / 4.75K * 10
	// simplified: Vout = LSB * 0.335
	// To prevent an overflow and the loss of the first decimal place a
	// long integer is needed for calculation.
	
	voltage_long = (unsigned long)lsb * 335;
	voltage_long = voltage_long / 100;
	
	voltage_int = voltage_long;
	return voltage_int;
}

unsigned int lsb_to_ac_voltage_adc(unsigned int lsb)
{
	unsigned long voltage_long;
	unsigned int voltage_int;

	// The input AC voltage is isolated and translated into 0...2.5V when input RMS voltage goes from 0 to 300Vac
	// This formula is used to calculate the Vac voltage from the LSBs:
	// Vac = LSB * (2.5/1023) * (300/2.5)
	// simplified: Vac = LSB * 0.293.
	// To prevent an overflow and the loss of the first decimal place a
	// long integer is needed for calculation.

	voltage_long = (unsigned long)lsb * 293;
	voltage_long = voltage_long / 1000;

	voltage_int = voltage_long;
	return voltage_int;
}


// Conversion of the sampled digital value into current.
// The value has three digits, the last represents hundred milliamps.
// (5650 means 56.50A)
unsigned int lsb_to_current_adc(unsigned int lsb)
{
	unsigned long current_long;
	unsigned int current_int;
	// 33mV equals 1A
	// This formula is used to calculate the output current from the LSBs:
	// Iout = LSB * 2.5/1023 / 0.033 * 10
	// simplified: Iout = LSB * 0.741 
	// To prevent an overflow and the loss of the first decimal place a
	// long integer is needed for calculation.
	current_long = (unsigned long)lsb * 741;
	current_long = current_long / 100;
	current_int = current_long;
	return current_int;
}



// Run the AD converter and calculate the temperature of the 3 NTCs
// and the output voltage and current.
void get_analog_inputs(void)
{
	// variables used for reading the analog inputs
	static volatile unsigned int analog_counter=0;
	static volatile unsigned int temperature_new=0, temperature_old=0;
	static volatile unsigned int ac_voltage_new=0, ac_voltage_old=0;
	static volatile unsigned int analog_voltage_new=0, analog_voltage_old=0;
	static volatile unsigned int analog_current_new=0, analog_current_old=0;
	unsigned int lsb_ac_voltage, lsb_voltage_output, lsb_current_output;
	unsigned int calculation_lsb;
	
	// Filter constant (Linear Shift-Invariant)
	const unsigned int analog_averaging = 8;
	
	// read ADC results
	// Skip if AD-conversion is not yet finished.
	// This will never happen due to the short conversion time.
	while (ADC10CTL1 & ADC10BUSY);
	
	switch(analog_counter)
	{
		case 0:	temperature_old = temperature_new;	// save last value
				temperature_new = ADC10MEM;			// read ADC memory
				
				// calculate new value and write to global variable in degree
				calculation_lsb = temperature_old * (analog_averaging - 1)/analog_averaging
				+ (temperature_new / analog_averaging);
				
				temperature_plus_50 = lsb_to_temperature_adc(calculation_lsb);
					
				analog_counter = 15;			// shift channel
				
				// stop ADC to change the input channel
				clear_bit(ADC10CTL0, ENC);
				
				//set channel to A15
				set_bit(ADC10CTL1, INCH3);
				set_bit(ADC10CTL1, INCH2);
				set_bit(ADC10CTL1, INCH1);
				set_bit(ADC10CTL1, INCH0);
					
				// enable analog input for A15
				clear_bit(ADC10AE0, A7_BIT);
				clear_bit(ADC10AE0, A6_BIT);
				set_bit(ADC10AE1, A15_BIT);
				clear_bit(ADC10AE0, A0_BIT);
				
				// enable and start conversion
				set_bit(ADC10CTL0, ENC);
   				set_bit(ADC10CTL0, ADC10SC);
   				break;

		case 15:
				ac_voltage_old = ac_voltage_new;	// save last value
				ac_voltage_new = ADC10MEM;				// read ADC memory
				
				// calculate new value and write to global variable in volt
				lsb_ac_voltage = ac_voltage_old * (analog_averaging - 1)/analog_averaging
				+ (ac_voltage_new / analog_averaging);
				
				voltage_ac = lsb_to_ac_voltage_adc(lsb_ac_voltage);
				analog_counter = 6;			// shift channel
				
				// stop ADC to change the input channel
				clear_bit(ADC10CTL0, ENC);
				
				//set channel to A6
				clear_bit(ADC10CTL1, INCH3);
				set_bit(ADC10CTL1, INCH2);
				set_bit(ADC10CTL1, INCH1);
				clear_bit(ADC10CTL1, INCH0);

				// enable analog input for A6
				clear_bit(ADC10AE0, A7_BIT);
				set_bit(ADC10AE0, A6_BIT);
				clear_bit(ADC10AE1, A15_BIT);
				clear_bit(ADC10AE0, A0_BIT);
				
				// enable and start conversion
				set_bit(ADC10CTL0, ENC);
   				set_bit(ADC10CTL0, ADC10SC);
   				break;

		case 6:
				analog_voltage_old = analog_voltage_new;	// save last value
				analog_voltage_new = ADC10MEM;				// read ADC memory
				
				// calculate new value and write to global variable in volt
				lsb_voltage_output =
				analog_voltage_old * (analog_averaging - 1)/analog_averaging
				+ (analog_voltage_new / analog_averaging);
				
				voltage_output = lsb_to_voltage_adc(lsb_voltage_output);
				analog_counter = 7;			// shift channel
				
				// stop ADC to change the input channel
				clear_bit(ADC10CTL0, ENC);
				
				//set channel to A7
				clear_bit(ADC10CTL1, INCH3);
				set_bit(ADC10CTL1, INCH2);
				set_bit(ADC10CTL1, INCH1);
				set_bit(ADC10CTL1, INCH0);
					
				// enable analog input for A7
				set_bit(ADC10AE0, A7_BIT);
				clear_bit(ADC10AE0, A6_BIT);
				clear_bit(ADC10AE1, A15_BIT);
				clear_bit(ADC10AE0, A0_BIT);
				
				// enable and start conversion
				set_bit(ADC10CTL0, ENC);
   				set_bit(ADC10CTL0, ADC10SC);
   				break;
   					
		case 7:	analog_current_old = analog_current_new;	// save last value
				analog_current_new = ADC10MEM;				// read ADC memory
				
				// calculate new value and write to global variable in ampere
				lsb_current_output =
				analog_current_old * (analog_averaging - 1)/analog_averaging
				+ (analog_current_new / analog_averaging);
				
				current_output = lsb_to_current_adc(lsb_current_output);
				
				analog_counter = 0;			// shift channel
				
				// stop ADC to change the input channel
				clear_bit(ADC10CTL0, ENC);
				
				//set channel to A0
				clear_bit(ADC10CTL1, INCH3);
				clear_bit(ADC10CTL1, INCH2);
				clear_bit(ADC10CTL1, INCH1);
				clear_bit(ADC10CTL1, INCH0);
				
				// enable analog input for A0
				clear_bit(ADC10AE0, A7_BIT);
				clear_bit(ADC10AE0, A6_BIT);
				clear_bit(ADC10AE1, A15_BIT);
				set_bit(ADC10AE0, A0_BIT);
				
				// enable and start conversion
				set_bit(ADC10CTL0, ENC);
				set_bit(ADC10CTL0, ADC10SC);
				break;
			
	default:	break;
	}	
}
